=begin
  Copyright 2010 (c), TIG
  All Rights Reserved.
  THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED 
  WARRANTIES,INCLUDING,WITHOUT LIMITATION,THE IMPLIED WARRANTIES OF 
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
###
  manifold.rb
###
  It looks at the contents of a selected group and tries to fix it so it 
  is 'manifold' for 3D-printing etc.
  The group needs to be a 3D shape which is fully surfaced without 
  internal divisions or intraneous/extraneous geometry/entities...
  When run...
  It removes 'non-face/non-edge' entities.
  It removes 'disconnected geometry'.
  It removes 'unfaced-edges'.
  It runs an initial 'health-check' and reports.
  It removes face 'flaps'.
  It heals 'holes'.
  It removes 'internal faces'.
  It orients all faces consistently to face 'outwards'.
  It erases 'coplanar edges' [optional].
  It 'triangulates' all of the faces [optional].
  It highlights major errors.
###
Usage:
  Select a group that is at least a bit 'manifold' [ give us a break ! ]
  Select the tool from the Plugins Menu > 'Manifold'
  OR type 'manifold' in the Ruby Console
  OR activate the 'Manifold' toolbar from View > Toolbars and click on 
  its button.
  It then processes the group, making a modified copy of it set off to 
  one side.
  If it has errors then a nested group is overlaid that has these 
  errors highlighted - red for removals, green for healing.
  This error-group has a 'text-handle' so you can easily select it to 
  move it away or erase it later...
  There is a closing dialog which either tells there were no major 
  errors [although it might have removed non-face/edge entities or 
  oriented faces and done some triangulation] or it reports the number 
  of errors.
  If X-ray mode is off it asks you if you want to change to X-ray Mode 
  to look inside the group to see what has been affected.
  On Yes/No the tool closes, with the new 'manifold' group now selected.
  Otherwise OK to end, with the new 'manifold' group now selected.

Donations:
   Are welcome [by PayPal], please use 'TIGdonations.htm' in the 
   ../Plugins/TIGtools/ folder.
   
Version:
   1.0 20100124 First release.
   1.1 20100124 Menu typo glitch fixed.
   1.2 20100125 Ticker added. Better processing for inner faces.
                Three-faced-edged external-faces now usually kept.
                FR lingvo updated by Pilou
   1.3 20100126 Trapped near colinear/planar points healing crashes.
                Very small edges now 'erased' by combining vertices.
                Orienting faces is now 'foolproof'.
                Unhealed or Failed Fixes now reported - with failed 
                edges shown as . or -.- clines in Error Group.
                Now only asks Y/N change to Xray mode if not on already.
   1.4 20100127 Minor zero length vector glitches trapped.
                Better inner 'flap' removal algorithm.
   1.5 20100129 Mesh is now intersected with self before processing.
                The flattenUVQ glitch addressed.
                Small line healing length now 0.5mm.
                New group's edges NOT hidden/smoothed/softened/layer0...
                Ticker made verbose.  Test loops now break if 'ok'.
                Cline 'errors' shown with cpoint ends for clarity.
                ES lingvo updated by Defisto.
                Updated all lingvo files for new strings.
   1.6 20100130 Face making error fixed, 3-edge faces removal improved.
                Sub-groups/Instances now within Group removed & reported.
                Error reporting improved.
   1.7 20100131 Inital quick 'Health-Check' introduced: Y/N to continue.
   1.8 20100131 Final triangulation is now optional - Y/N.
   1.9 20100031 Health-Check now traps for 'No Geometry' in group.
   2.0 20100201 Very small face glitch fixed during health-check.
   2.1 20100209 Optional 'erase coplanar edges'. Pilou updated FR lingvo.
   2.2 20100222 Tooltips and other Strings now deBabelized properly.
                FR lingvo updated by Didier Bur.
                
TO DO: Get three-faced-edges external-faces to be retained in 'pretzels'.
=end

require 'sketchup.rb'
require 'deBabelizer.rb'

class Manifold

def db(string)
  dir=File.dirname(__FILE__)+"/TIGtools"
  toolname="manifold" 
  locale=Sketchup.get_locale.upcase
  path=dir+"/"+toolname+locale+".lingvo"
  if not File.exist?(path)
    return string
  else
    deBabelizer(string,path)
  end 
end#def

def Manifold::db(string)
  dir=File.dirname(__FILE__)+"/TIGtools"
  toolname="manifold" 
  locale=Sketchup.get_locale.upcase
  path=dir+"/"+toolname+locale+".lingvo"
  if not File.exist?(path)
    return string
  else
    deBabelizer(string,path)
  end 
end#def

def activate
    @model=Sketchup.active_model
    @ents=@model.active_entities
    @ss=@model.selection
    @original_group=@ss[0]
    if not @original_group or not @original_group.class==Sketchup::Group
        UI.messagebox(db("Manifold: Select a GROUP BEFORE use."))
        Sketchup.send_action("selectSelectionTool:")
        return nil
    end#if
    @group=nil
    @error_group=nil
    @temp_error_group=nil
    @msg=""
    Sketchup::set_status_text(@msg)
    @removals=false
    @healed=false
    @failures=false
    if Sketchup.version[0,1].to_i > 6
	  @model.start_operation((db("Manifold")),true) ### ?
	else
	  @model.start_operation((db("Manifold")))
	end#if
    @done=false
    @going_on=false
    copy_group()
    remove_faceless_edges()
    remove_unconnected()
    health_check()
  if @going_on
    remove_small_edges()
    heal_holes()
    remove_flaps()
    remove_inner_faces()
    remove_loose_faces()
    erase_coplanar_edges()
    triangulate_faces()
    orient_faces()
    move_to_side()
    closing_tidy()
    closing_msg()
    @done=true
    deactivate(nil)
  end#if
end

def resume(view)
    Sketchup::set_status_text(@msg)
    view.invalidate
end

def deactivate(view=nil)
    if not @done
      @group.erase! if @group and @group.valid?
      @error_group.erase! if @error_group and @error_group.valid?
      @temp_error_group.erase! if @temp_error_group and @temp_error_group.valid?
    end#if
    view.invalidate if view
    @model.commit_operation
    Sketchup.send_action("selectSelectionTool:")
    return nil
end

def onCancel(reason,view)
    deactivate(view)
    Sketchup.send_action("selectSelectionTool:")
    return nil
end

def copy_group()
  @msg=(db("Manifold"));Sketchup::set_status_text(@msg)
  ### make the error group
  @error_group=@ents.add_group()
  @egents=@error_group.entities
  #@egents.add_cpoint(@original_group.bounds.max)
  #@egents.add_cpoint(@original_group.bounds.min)
  tr=Geom::Transformation.translation(ORIGIN.vector_to(@original_group.bounds.min))
  @error_group.transform!(tr)
  ###
  @group=@original_group.copy
  ### try to force ordering of gents
  @msg=(db("Manifold"))+"...";Sketchup::set_status_text(@msg)
  tgp=@ents.add_group(@group)
  @msg=(db("Checking Group Entities..."));Sketchup::set_status_text(@msg)
  @group.explode
  @group=tgp
  ### remove non-face/edge ents
  @gents=@group.entities
  ctr=0
  to_go=[]
  @gents.to_a.each{|e|
    @msg=(db("Checking Group Entities: "))+ctr.to_s;Sketchup::set_status_text(@msg);ctr+=1
    next if not e.valid?
    if e.class==Sketchup::Edge
      e.smooth=false
      e.soft=false
      e.hidden=false
      e.layer=nil
      e.material=nil
    elsif e.class==Sketchup::Face
      e.hidden=false
      e.layer=nil
      ###e.material=nil ### we'll keep face mats ?
    else
      clone_removed_into_error_group([e])
      to_go<< e
    end#if
  }
  @msg=(db("Copying Group..."));Sketchup::set_status_text(@msg)
  @gents.erase_entities(to_go)if to_go[0]
  ###
end

def remove_faceless_edges()
  @msg=(db("Removing Faceless Edges..."));Sketchup::set_status_text(@msg)
  to_go=[]
  @gents.to_a.each{|e|
    if e.valid? and e.class==Sketchup::Edge and e.faces.length==0
      clone_removed_into_error_group([e])### takes an array
      to_go<< e if e.valid? 
    end#if
  }
  @gents.erase_entities(to_go)if to_go[0]
end

def remove_unconnected()
  @msg=(db("Removing Unconnected Geometry..."));Sketchup::set_status_text(@msg)
  faces=[];@gents.each{|e|faces<<e if e.class==Sketchup::Face}
  hiface=faces[0]
  faces.each{|e|hiface=e if e.bounds.max.z > hiface.bounds.max.z}
  left_overs=@gents.to_a-hiface.all_connected if hiface and @gents.to_a
  @gents.to_a.each{|e|left_overs<<e if not e.class==Sketchup::Face and not e.class==Sketchup::Edge}
  clone_removed_into_error_group(left_overs)if left_overs
  left_overs.each{|e|e.erase! if e.valid?}if left_overs
end

def health_check()
  @msg=(db("Initial Health-Check..."));Sketchup::set_status_text(@msg)
  edges=[];@gents.each{|e|edges<<e if e.class==Sketchup::Edge}
  faces1=[];faces3=[];shorts=[]
  edges.each{|e|
    faces1 << e.faces if e.faces.length==1 ### an array of faces
    faces3 << e.faces if e.faces.length==3 ### an array of faces
    shorts << e if e.length<=0.5.mm ### ?????????????
  }
  @temp_error_group=@error_group.copy
  txt0="";txt3="";txtS=""
  if faces1[0]
    faces1.each{|faces|clone_into_temp_hole_error_group(faces)}
    txt0=faces1.length.to_s+(db(" Edges form potential Holes [=Cyan]."))+"\n"
  end#if
  if faces3[0]
    faces3.each{|faces|clone_removed_into_temp_error_group(faces)}
    txt3=faces3.length.to_s+(db(" Edges form potential Internal Faces [=Magenta]."))+"\n"
  end#if
  if shorts[0]
    ###???
    txtS=shorts.length.to_s+(db(" Edges are VERY 'short' and need Fixing."))+"\n"
  end#if
  xray=@model.rendering_options["ModelTransparency"]
  ###
  if txt0 !="" or txt3 !="" or txtS !=""
    msg=@msg+"\n\n"+txt3+txt0+txtS
    @model.rendering_options["ModelTransparency"]=true
    @group.hidden=true
    @original_group.hidden=true
  elsif edges.length==0
    @temp_error_group.erase! if @temp_error_group and @temp_error_group.valid?
    @model.rendering_options["ModelTransparency"]=true
    @original_group.hidden=true
    msg=@msg+"\n\n"+(db("There is NO Geometry to Check !"))
    UI.messagebox(msg)
    @original_group.hidden=false
    @group.erase! if @group and @group.valid?
    @model.rendering_options["ModelTransparency"]=xray
    @going_on=false
    deactivate(nil)
    Sketchup.send_action("selectSelectionTool:")
    return nil
  else
    msg=@msg+"\n\n"+(db("No Initial Problems found."))
  end#if
  msg=msg+"\n\n"+(db("Continue ?"))
  ###
  UI.beep
  reply=UI.messagebox(msg,MB_YESNO)### 6=YES 7=NO
  ###
  if reply==7 ###=no
    @group.hidden=false
    @original_group.hidden=false
    @model.rendering_options["ModelTransparency"]=xray
    @temp_error_group.erase!
    @going_on=false
    deactivate(nil)
    Sketchup.send_action("selectSelectionTool:")
    return nil
  else ###continuing
    @group.hidden=false
    @original_group.hidden=false
    @model.rendering_options["ModelTransparency"]=xray
    @temp_error_group.erase!
    @going_on=true
  end#if
end

def remove_small_edges()
  @msg=(db("Searching for Small Edges: "));Sketchup::set_status_text(@msg)
  edges=[];@gents.each{|e|edges<<e if e.class==Sketchup::Edge}
  ctr=0
  edges.each{|edge|
    if edge.valid? and edge.length<=0.5.mm ### ?????????????
      ### it's so small we should 'erase' it by combining its vertices
      @msg=(db("Searching for Small Edges: "))+ctr.to_s;Sketchup::set_status_text(@msg);ctr+=1
      clone_removed_into_error_group([edge])
      v0=edge.start;v1=edge.end
      p0=v0.position;p1=v1.position
      tr=Geom::Transformation.translation(p1.vector_to(p0))
      @gents.transform_entities(tr,v1)
    end#if
  }
end

def clone_removed_into_error_group(ents=[])
  @removals=true
  ents.each{|e|
    if e and e.valid? and e.class==Sketchup::Face
      pts=[];e.outer_loop.vertices.each{|v|pts<<v.position.to_a}
      pts.uniq!
     begin
      fa=@egents.add_face(pts)if pts[2]
      fa.material="red" if fa
      fa.back_material="red" if fa
     rescue
      ###
     end
    end#if
    if e and e.valid? and e.class==Sketchup::Edge
     begin
      ln=@egents.add_line(e.start.position,e.end.position)
      ln.material="red" if ln
     rescue
      ###
     end
    end#if
    if e and e.valid? and e.class==Sketchup::Group
      tr=e.transformation
      tgp=@egents.add_group()
      tgpents=tgp.entities
      ins=tgpents.add_instance(e.real_parent,tr)
      ins.explode
      tents=tgpents.to_a
      tents.each{|t|t.material="red"}
      tgp.material="red"
    end#if
    if e and e.valid? and e.class==Sketchup::ComponentInstance
      tr=e.transformation
      tgp=@egents.add_group()
      tgpents=tgp.entities
      ins=tgpents.add_instance(e.definition,tr)
      ins.explode
      tents=tgpents.to_a
      tents.each{|t|t.material="red"}
      tgp.material="red"
    end#if
  }
end

def clone_removed_into_temp_error_group(ents=[])
  tents=@temp_error_group.entities
  ents.each{|e|
    if e and e.valid? and e.class==Sketchup::Face
      pts=[];e.outer_loop.vertices.each{|v|pts<<v.position.to_a}
      pts.uniq!
     begin
      fa=tents.add_face(pts)if pts[2]
      fa.material="magenta" if fa
      fa.back_material="magenta" if fa
     rescue
      ###
     end
    end#if
    if e and e.valid? and e.class==Sketchup::Edge
     begin
      ln=tents.add_line(e.start.position,e.end.position)
      ln.material="magenta" if ln
     rescue
      ###
     end
    end#if
  }
end
def clone_into_temp_hole_error_group(ents=[])
  tents=@temp_error_group.entities
  ents.each{|e|
    if e and e.valid? and e.class==Sketchup::Face
      pts=[];e.outer_loop.vertices.each{|v|pts<<v.position.to_a}
      pts.uniq!
     begin
      fa=tents.add_face(pts)if pts[2]
      fa.material="cyan" if fa
      fa.back_material="cyan" if fa
     rescue
      ###
     end
    end#if
    if e and e.valid? and e.class==Sketchup::Edge
     begin
      ln=tents.add_line(e.start.position,e.end.position)
      ln.material="cyan" if ln
     rescue
      ###
     end
    end#if
  }
end

class Sketchup::Group
       # Some times the group.entities.parent refer to the wrong definition. This method checks for this error and locates the correct parent definition.
       def real_parent
          if self.entities.parent.instances.include?(self)
             return self.entities.parent
          else
             Sketchup.active_model.definitions.each { |definition|
                return definition if definition.instances.include?(self)
             }
          end
          return nil # Should not happen.
       end
    end

def clone_healed_into_error_group(ents=[])
  @healed=true
  ents.each{|e|
    if e and e.valid? and e.class==Sketchup::Face
      pts=[];e.outer_loop.vertices.each{|v|pts<<v.position.to_a}
      pts.uniq!
     begin
      fa=@egents.add_face(pts)if pts[2]
      fa.material="green" if fa
      fa.back_material="green" if fa
     rescue
      ###
     end
    end#if
    if e and e.valid? and e.class==Sketchup::Edge
     begin
      ln=@egents.add_line(e.start.position,e.end.position)
      ln.material="green" if ln
     rescue
      ###
     end
    end#if
  }
end

def heal_holes()
  @msg=(db("Intersecting..."));Sketchup::set_status_text(@msg)
  ### first intersect group with self
  @gents.intersect_with(true,@group.transformation,@group,@group.transformation,true,@group)
  ###
  @msg=(db("Searching for Holes: "));Sketchup::set_status_text(@msg)
  ###
  edges=[];@gents.each{|e|edges<<e if e.class==Sketchup::Edge}
  cedge0=nil;cedge1=nil
  ok=true ###
(edges.length).times do |i|
  edges=[];@gents.each{|e|edges<<e if e.class==Sketchup::Edge}
  cedge0=nil;cedge1=nil
  @msg=(db("Searching for Holes: "))+i.to_s;Sketchup::set_status_text(@msg)
  ctr=0
  edges.each{|edge|
    Sketchup::set_status_text(@msg+"~"+ctr.to_s);ctr+=1
    cedge0=nil;cedge1=nil
    if edge.faces.length==1 ### possible 'flap'
      edge.start.edges.each{|e|
        if not e==edge and e.faces.length==1
          cedge0=e
        end#if
      }
      edge.end.edges.each{|ee|
        if not ee==edge and  ee.faces.length==1
          cedge1=ee
        end#if
      }
    end#if
    ### we now have the edges connected to ends of the this one-faced edge
    if cedge0 and cedge1
      if edge.line[1].angle_between(cedge0.line[1]) <  edge.line[1].angle_between(cedge1.line[1])
        cedge=cedge0
      else
        cedge=cedge1
      end#if
    elsif cedge0
      cedge=cedge0
    elsif cedge1
      cedge=cedge1
    else ### no cedges
      cedge=nil
    end#if
    if cedge and (edge.line[1] != cedge.line[1] or  edge.line[1] != cedge.line[1].reverse!) and (edge.line[1].angle_between(cedge.line[1])!=0.0.degrees or edge.line[1].angle_between(cedge.line[1])!=180.0.degrees) ### not colinear so we try to heal the hole
      if edge.start==cedge.start
        vert2=cedge.end
      elsif edge.start==cedge.end
        vert2=cedge.start
      elsif edge.end==cedge.start
        vert2=cedge.end
      elsif edge.end==cedge.end
        vert2=cedge.start
      end#if
      begin
        face=@gents.add_face(edge.start.position,edge.end.position,vert2.position)
        clone_healed_into_error_group([face]+face.edges)if face
        ok=false ###
      rescue
        #puts "???????"
        #puts edge.start.position;puts edge.end.position;puts vert2.position
        #puts "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
      end
    end#if
  }
  break if ok ###
 end#do
end

def remove_flaps()
  edges=[];@gents.each{|e|edges<<e if e.class==Sketchup::Edge}
  @msg=(db("Removing Flaps..."));Sketchup::set_status_text(@msg)
  edges.each{|edge|
    if edge.valid? and edge.faces.length==1 ### single faced ### see if 'flap'
      ### erase edge and it's face etc IF 'flapped'
      face=edge.faces[0]
      to_go=[];face.edges.each{|e|to_go<<e if e.faces.length==1}
      if to_go[1]
        clone_removed_into_error_group([face])
        face.edges.each{|e|
          clone_removed_into_error_group([e])
          e.erase! if e.valid? and e.faces.length==1
        }
      end#if
      ###
     if edge.valid?
      flap=true
      edge.vertices[0].edges.each{|e|
        if not e==edge and e.faces.length==1
          flap=false
        end#if
      }
      edge.vertices[1].edges.each{|e|
        if not e==edge and e.faces.length==1
          flap=false
        end#if
      }
      if flap
        clone_removed_into_error_group([edge]+[edge.faces[0]])
        edge.erase! if edge.valid? 
      else ### heal
        heal_holes()
      end#if
      ###
     end#if
   end#if
  }
end

def remove_inner_faces()
  @msg=(db("Searching for Inner Faces: "));Sketchup::set_status_text(@msg)
  faces=[];@gents.each{|e|faces<<e if e.class==Sketchup::Face}
  faces.reverse! ## so do last first
  faces_to_go=[]
  to_go=false
  ok=true ###
 faces.length.times do |i|
  @msg=(db("Searching for Inner Faces: "))+i.to_s;Sketchup::set_status_text(@msg)
  to_go=false
  ctr=0
  faces.each{|face|
    Sketchup::set_status_text(@msg+"~"+ctr.to_s);ctr+=1
    to_go=true
    face.outer_loop.edges.each{|e|
      if e.valid? and e.faces.length<=2
        to_go=false
        break
      end#if
    }### we need to trap for external faces that have all edges with 3 faces !!!
      ### raytest fails with 'pretzels'... so need to test more... 
    if to_go and raytest(face) #################
      faces_to_go << face
      ok=false ###
    end#if
    ###
  }
  break if ok ###
 end#times
  ctrr=0
  faces_to_go.uniq.each{|face|
    @msg=(db("Removing Inner Faces: "))+ctrr.to_s;Sketchup::set_status_text(@msg);ctrr+=1
    if face.valid?
      clone_removed_into_error_group([face]+face.edges)
      face.erase!
    end#if
  }if faces_to_go.uniq
#=begin
  ### other possibles
  faces=[];@gents.each{|e|faces<<e if e.class==Sketchup::Face}
  faces.reverse!
  faces3=[]
  faces.each{|face|
    face.outer_loop.edges.each{|e|
      next if not e.valid? 
      if e.faces.length>2
        faces3<<face
        break
      end#if
    }
  }
  faces_to_go=[]
  ctr=0
  faces3.each{|face|
    @msg=(db("Searching for Inner Faces: "))+ctr.to_s;Sketchup::set_status_text(@msg);ctr+=1
    next if not face.valid?
    if raytest(face) #################
      faces_to_go << face
      ok=false ###
    end#if
    ###
    remove_faceless_edges()
    remove_unconnected()
    remove_flaps()
  }
  ctri=0
  faces_to_go.uniq.each{|face|
    @msg=(db("Removing Inner Faces: "))+ctri.to_s;Sketchup::set_status_text(@msg);ctri+=1
    if face.valid?
      clone_removed_into_error_group([face]+face.edges)
      face.erase!
    end#if
  }if faces_to_go.uniq
#=end
  remove_faceless_edges()
  remove_unconnected()
  remove_flaps()
  ###
end

def raytest(face)
  p0=face.outer_loop.vertices[0].position
  p1=face.outer_loop.vertices[1].position
  p2=face.outer_loop.vertices[2].position
  vm=p0.vector_to(p1)
  return false if vm.length==0.0
  pm=p0.offset(vm,p0.distance(p1)/2)
  vx=pm.vector_to(p2)
  return false if vx.length==0.0
  px=pm.offset(vx,pm.distance(p2)/2)
  if rayt=@model.raytest(px.transform!(@group.transformation),face.normal)
    hit=rayt.flatten!
  else
    hit=nil
  end#if
  if hit and (hit.include?(@group) or hit.include?(@original_group) or hit.include?(@error_group))
    ### test that hit is true - is face on outer edge of the volune ?
    return true #if is_inner?(face) ##################
  else
    return false
  end#if
  return false
end

=begin
def is_inner?(face=nil)
  return false if not face
  bb=face.bounds
  center=bb.center.transform!(@group.transformation)
  slice=@ents.add_group()
  sents=slice.entities
  if face.normal.z.abs==1.0
    normal=X_AXIS
  else
    normal=Z_AXIS
  end#if
  radius=@group.bounds.diagonal
  circedges=sents.add_circle(center,normal,radius,12)
  sents.add_face(circedges)
  sents.intersect_with(true,slice.transformation,slice,slice.transformation,true,@group)
  sents.length.times{|i|sents.intersect_with(true,slice.transformation,slice,slice.transformation,true,sents[i])}
  sents.erase_entities(circedges)
  edges=[];sents.each{|e|edges<<e if e.class==Sketchup::Edge}
  edges2=[];edges.each{|e|edges2<<e if e.faces.length>1}
  hit=false
  edges2.each{|e|
    ct=e.start.position.offset!(e.line[1],0.1.mm)
    if face.classify_point(ct)>=1 and face.classify_point(ct)<=7
      hit=true
      break
    end#if
  }
  slice.erase! if slice.valid?
  if hit
    return true
  else
    return false
  end#if
end
=end

def remove_loose_faces()
  faces=[];@gents.each{|e|faces<<e if e.class==Sketchup::Face}
  faces.reverse!
  faces_to_go=[]
  breaker=true
 faces.length.times do |i|
  faces=[];@gents.each{|e|faces<<e if e.class==Sketchup::Face}
  @msgi=(db("Finding Loose Faces: "))+i.to_s
  ctr=0
  faces.each{|face|
    @msg=@msgi+"~"+ctr.to_s;Sketchup::set_status_text(@msg);ctr+=1
    face.edges.each{|e|
      if e.faces.length==1
        ### might be loose edge in inside 
        ### OR on outer skin because it's a 'pretzel' ?
        internal=false
        face.edges.each{|ee|
          if ee.faces.length>2
            internal=true
            break
          end#if
        }
        faces_to_go<<face if internal
        ### otherwise it's external skin face
        break
      end#if
    }
  }
  faces_to_go.uniq.each{|face|
    if face.valid?
      clone_removed_into_error_group([face]+face.edges)
      face.erase!
      breaker=false
    end#if
  }if faces_to_go.uniq
  remove_faceless_edges()
  breaker if breaker
 end#times
  ###
end

class Sketchup::Face
 def orient_manifold_faces
  dir=File.dirname(__FILE__)+"/TIGtools"
  toolname="manifold" 
  locale=Sketchup.get_locale.upcase
  path=dir+"/"+toolname+locale+".lingvo"
    @connected_faces=[]
	self.all_connected.each{|e|
	  if e.class==Sketchup::Face
		e.edges.each{|edge|
		  if edge.faces[1]
            @connected_faces<<e
            break
          end#if
        }
	  end#if
	}
    @connected_faces=[self] + @connected_faces 
    @connected_faces.uniq!
	@awaiting_faces=@connected_faces
    @processed_faces=[self]
    @done_faces=[]
    msg=(deBabelizer("Manifold: Orienting Faces",path))
    ###
    longstop=256;count=0
	while @awaiting_faces[0]
      msg=msg+"."
	  @processed_faces.each{|face|
        if not @done_faces.include?(face)
	      Sketchup::set_status_text(msg,SB_PROMPT)
		  @face=face
          face_flip
        end#if
	  }
      count+=1
      if count==longstop
       puts(deBabelizer("Manifold: orient faces aborted.",path))
       break
     end#if
    end#while
 end#def
 def face_flip
    @awaiting_faces=@awaiting_faces-[@face]
    @face.edges.each{|edge|
      rev1=edge.reversed_in?(@face)
      @common_faces=edge.faces-[@face]
      @common_faces.each{|face|
	    rev2=edge.reversed_in?(face)
        face.reverse! if @awaiting_faces.include?(face) and rev1==rev2
	    @awaiting_faces=@awaiting_faces-[face]
	    @processed_faces<<face
	  }
    }
    @done_faces<<@face
 end#def
end#class

def orient_faces()
  @msg=(db("Orienting Faces..."));Sketchup::set_status_text(@msg)
  the_face=nil
  bb=@group.bounds
  maxz=bb.min.z
  @gents.to_a.each{|face|
    if face.class==Sketchup::Face
      fb=face.bounds
      if fb.max.z > maxz and face.normal.z != 0.0
        the_face=face
        maxz=fb.max.z
      end#if
    end#if
  }### see if face looks up
  if the_face
    the_face.reverse! if the_face.normal.z<0.0
    the_face.orient_manifold_faces
  else
    UI.beep
    puts (db("Manifold: orient faces aborted."))
  end#if
end

def triangulateFace(face)
 def flattenUVQ(uvq)# Get UV coordinates from UVQ matrix. (TT)
   return Geom::Point3d.new(uvq.x/uvq.z,uvq.y/uvq.z,1.0)
 end
 ### if already a triangle leave alone...
 if face.vertices[3]
  mesh=face.mesh(3)
  ents=face.parent.entities
  mat=face.material
  bac=face.back_material
   # (TT) # Instead of sampling vertices we sample points of the face's plane.
   # This is because we need four points, and some times a face only has three vertices.
   # And we also get the wrong result if any of the first four vertices are co-linear.
   samples = []
   samples << face.vertices[0].position               # 0,0 | Origin
   samples << samples[0].offset(face.normal.axes.x) # 1,0 | Offset Origin in X
   samples << samples[0].offset(face.normal.axes.y) # 0,1 | Offset Origin in Y
   samples << samples[1].offset(face.normal.axes.y) # 1,1 | Offset X in Y
   ### get uvs
   tw=Sketchup.create_texture_writer
   # (TT)
   uv_helper=face.get_UVHelper(true,true,tw) # (TT) Only need one UV Helper
   xyz = []
   uv_f = []
   uv_b = []
   samples.each { |position|
      # XYZ 3D coordinates
      xyz << position
      # UV 2D coordinates
      # Front
      uvq = uv_helper.get_front_UVQ(position)
      uv_f << flattenUVQ(uvq)
      # Back
      uvq = uv_helper.get_back_UVQ(position)
      uv_b << flattenUVQ(uvq)
   }
  ###
  grp=ents.add_group
  grp.entities.fill_from_mesh(mesh, true, 0)
  # (TT) Adds the mesh without soft/smooth edges. (and faster)
  grp.entities.each{|e|
    next unless e.class==Sketchup::Face
    if mat # (TT) # Position texture.
      pts = []
      (0..3).each { |i|
         pts << xyz[i]
         pts << uv_f[i]
      }
        e.position_material(mat,pts, true)
    end#if
    if bac # (TT) # Position texture.
      pts = []
      (0..3).each { |i|
         pts << xyz[i]
         pts << uv_b[i]
      }
       e.position_material(bac, pts, false)
    end#if
  }
  face.erase! if face.valid? 
  grp.explode
 end#if
end#def

def erase_coplanar_edges()
  @msg=(db("Erase Coplanar Edges ?"));Sketchup::set_status_text(@msg)
  UI.beep
  reply=UI.messagebox(@msg,MB_YESNO)### 6=YES 7=NO
 if reply==6
   counter=0
    4.times do ### to make sure we got all of them !
     @gents.to_a.each{|e|
      if e.valid? and e.class==Sketchup::Edge
       if not e.faces[0]
         e.erase!
         counter+=1
         @msg=((db("Coplanar Edges Erased= "))+counter.to_s)
         Sketchup::set_status_text(@msg)
       elsif e.faces.length==2 and e.faces[0].normal.dot(e.faces[1].normal)> 0.99999999999 ####
         e.erase!
         counter+=1
         @msg=((db("Coplanar Edges Erased= "))+counter.to_s)
         Sketchup::set_status_text(@msg)
       end#if
      end#if
     }
    end#times
 end#if
end

def triangulate_faces()
  @msg=(db("Triangulate Faces ?"));Sketchup::set_status_text(@msg)
  UI.beep
  reply=UI.messagebox(@msg,MB_YESNO)### 6=YES 7=NO
 if reply==6
  ctr=0
  @gents.to_a.each{|e|
    @msg=(db("Triangulating Faces: "))+ctr.to_s;Sketchup::set_status_text(@msg);ctr+=1
    triangulateFace(e)if e.valid? and e.class==Sketchup::Face
  }
 end#if
end

def move_to_side()
  bb=@original_group.bounds
  width=bb.width/1
  tranf=@group.transformation.to_a
  tranf[12]=(bb.max.x+width)
  @group.transformation=tranf
  tranf=@error_group.transformation.to_a
  tranf[12]=(bb.max.x+width)
  @error_group.transformation=tranf
end

def closing_tidy()
  Sketchup::set_status_text("")
  point=[0,0,0]
  bb=@group.bounds
  tr=Geom::Transformation.new(bb.max.transform!(@group.transformation.inverse))
  point.transform!(tr)
  string=(db("ERROR GROUP:"))+"\n"
  string=string+(db("Red=Removed"))+"\n" if @removals
  string=string+(db("Green=Healed"))+"\n" if @healed
  vector=[1,1,1]
  #@egents.to_a.each{|e|e.erase! if e.class==Sketchup::ConstructionPoint}
  txt=@egents.add_text(string,point,vector)
  txt.display_leader=false
  if @egents.to_a.length>1
    @errors=true
  else
    @errors=false
    @error_group.erase!
  end#if
  edges=[];@gents.each{|e|edges<<e if e.class==Sketchup::Edge}
  @fail_edge1=false
  edges.each{|e|
    if e.faces.length==1
      @fail_edge1=true
      cl=@egents.add_cline(e.start.position,e.end.position)
      cl.stipple="." if cl
      @egents.add_cpoint(e.start.position)
      @egents.add_cpoint(e.end.position)
      @failures=true
    end#if
  }
  @fail_edge3=false
  edges.each{|e|
    if e.faces.length>2
      @fail_edge3=true
      cl=@egents.add_cline(e.start.position,e.end.position)
      cl.stipple="-.-" if cl
      @egents.add_cpoint(e.start.position)
      @egents.add_cpoint(e.end.position)
      @failures=true
    end#if
  }
  if @failures ### add extra error text
    string=string+(db(".../-.-=Failed"))
    txt.text=string
  end#if
end

def closing_msg()
  @ss.clear
  @ss.add(@group)
  msg=(db("Manifold:"))+"\n\n"
  if not @errors
    msg=msg+(db("No Removal or Healing was required."))
    msg=msg+"\n"+(db("Faces oriented and triangulated."))
    UI.messagebox(msg)
  else
    msg=msg+((@egents.to_a.length)-1).to_s
    msg=msg+(db(" Errors were fixed."))
    msg=msg+"\n"+(db("Removed Items are shown red.")) if @removals
    msg=msg+"\n"+(db("Healed Items are shown green.")) if @healed

    msg=msg+"\n"+(db("Faces oriented and triangulated."))
    msg=msg+"\n\n"+(db("Some Holes could NOT be Healed !"))+(db(" ..... =Failed"))if @fail_edge1
    msg=msg+"\n\n"+(db("A fully Manifolded Mesh was NOT possible !"))+(db(" -.-.- =Failed"))if @fail_edge3
    if not @model.rendering_options["ModelTransparency"]
      msg=msg+"\n\n"+(db("Do you want to switch to X-ray Mode ?"))
      UI.beep
      reply=UI.messagebox(msg,MB_YESNO)### 6=YES 7=NO
    else
      UI.messagebox(msg)
      reply=7
    end#if
    @model.rendering_options["ModelTransparency"]=true if reply==6
  end#if
end
    
end#class -----------------------------------


### shortcut
def manifold()
    Sketchup.active_model.select_tool(Manifold.new)
end#def

### menu ################
if not file_loaded?(__FILE__)
    textstring=Manifold::db"Manifold"
    instructions=Manifold::db": Select a Group to be made 'Manifold', Run the Tool..."
    dir=File.dirname(__FILE__)+"/TIGtools"
    toolname="manifold"
    locale=Sketchup.get_locale.upcase
    path=dir+"/"+toolname+locale+".lingvo"
    if File.exist?(path)
      textstring=deBabelizer(textstring,path)
      instructions=deBabelizer(instructions,path)
    end#if
    cmd=UI::Command.new(textstring){manifold()}
    UI.menu("Plugins").add_item(textstring){manifold()}
    cmd.status_bar_text=textstring+instructions
    toolbar=UI::Toolbar.new(textstring)
    cmd.tooltip=textstring
    cmd.small_icon="TIGtools/manifold16x16.png"
    cmd.large_icon="TIGtools/manifold24x24.png"
    toolbar.add_item(cmd)
    toolbar.restore if toolbar.get_last_state==TB_VISIBLE
end
file_loaded(__FILE__)
